/**
* Nanotec Nanolib example
* Copyright (C) Nanotec GmbH & Co. KG - All Rights Reserved
*
* This product includes software developed by the
* Nanotec GmbH & Co. KG (http://www.nanotec.com/).
*
* The Nanolib interface headers and the examples source code provided are 
* licensed under the Creative Commons Attribution 4.0 Internaltional License. 
* To view a copy of this license, 
* visit https://creativecommons.org/licenses/by/4.0/ or send a letter to 
* Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
*
* The parts of the library provided in binary format are licensed under 
* the Creative Commons Attribution-NoDerivatives 4.0 International License. 
* To view a copy of this license, 
* visit http://creativecommons.org/licenses/by-nd/4.0/ or send a letter to 
* Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
*
* @file   BusFunctionsExample.java
*
* @brief  Definition of bus hardware specific functions
*
* @date   29-10-2024
*
* @author Michael Milbradt
*
*/
package com.nanotec.example.NanolibExample;

import com.nanotec.nanolib.ResultBusHwIds;
import com.nanotec.example.NanolibExample.MenuUtils.Context;
import com.nanotec.example.NanolibExample.MenuUtils.Menu;
import com.nanotec.nanolib.BusHardwareId;
import com.nanotec.nanolib.BusHardwareOptions;
import com.nanotec.nanolib.ResultVoid;
import com.nanotec.nanolib.DeviceHandle;

/**
 * @brief Container class for bus functions
 */
public class BusFunctionsExample {
    
    /**
    * @brief Retrieve list of available bus hardware and store to ctx.openableBusHardwareIds
    *
    * @param  ctx  the menu context
    * @return      void
    */
    public static void scanBusHardware(Context ctx) {
        ctx.waitForUserConfirmation = false;

        ResultBusHwIds result = ctx.nanolibAccessor.listAvailableBusHardware();
        if (result.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during bus scan: ", result.getError());
            return;
        }

        ctx.scannedBusHardwareIds = result.getResult();

        if (ctx.scannedBusHardwareIds.isEmpty()) {
            MenuUtils.handleErrorMessage(ctx, "No bus hardware found. Please check your cabling, driver and/or devices.", "");
            return;
        }

        ctx.openableBusHardwareIds = Menu.getOpenableBusHwIds(ctx);
    }

    /**
    * @brief Open the selected bus hardware (ctx.selectedOption)
    *
    * @param  ctx  the menu context
    * @return      void
    */
    public static void openBusHardware(Context ctx) {
        ctx.waitForUserConfirmation = false;

        if (ctx.openableBusHardwareIds.isEmpty()) {
            MenuUtils.handleErrorMessage(ctx, "No bus hardware available. Please do a scan first.", "");
            return;
        }

        int index = ctx.selectedOption;
        BusHardwareId busHwId = ctx.openableBusHardwareIds.get(index - 1);

        for (BusHardwareId openBusHwId : ctx.openBusHardwareIds) {
            if (openBusHwId.equals(busHwId)) {
                MenuUtils.handleErrorMessage(ctx, "Bus hardware " + busHwId.getName() + " already open.", "");
                return;
            }
        }

        BusHardwareOptions busHwOptions = MenuUtils.createBusHardwareOptions(busHwId);
        ResultVoid resultVoid = ctx.nanolibAccessor.openBusHardwareWithProtocol(busHwId, busHwOptions);
        if (resultVoid.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during openBusHardware: ", resultVoid.getError());
            return;
        }

        ctx.openBusHardwareIds.add(busHwId);
        ctx.openableBusHardwareIds = Menu.getOpenableBusHwIds(ctx);
    }

    /**
    * @brief Close the selected bus hardware (ctx.selectedOption)
    *
    * @param  ctx  the menu context
    * @return      void
    */
    public static void closeBusHardware(Context ctx) {
        ctx.waitForUserConfirmation = false;
        int index = ctx.selectedOption;

        if (ctx.openBusHardwareIds.isEmpty()) {
            MenuUtils.handleErrorMessage(ctx, "No open bus hardware found.", "");
            return;
        }

        BusHardwareId closeBusHardwareId = ctx.openBusHardwareIds.get(index - 1);

        // Remove connected device handles
        ctx.connectedDeviceHandles.removeIf(e -> closeBusHardwareId.equals(ctx.nanolibAccessor.getDeviceId(e).getResult().getBusHardwareId()));

        // Reset active device handle if it belongs to bus hardware to close
        if (ctx.nanolibAccessor.getDeviceId(ctx.activeDevice).getResult().getBusHardwareId().equals(closeBusHardwareId)) {
            ctx.activeDevice = new DeviceHandle();
        }

        // Remove available device IDs
        ctx.connectableDeviceIds.removeIf(e -> closeBusHardwareId.equals(e.getBusHardwareId()));
        ctx.scannedDeviceIds.removeIf(e -> closeBusHardwareId.equals(e.getBusHardwareId()));

        // Close the bus hardware
        ResultVoid resultVoid = ctx.nanolibAccessor.closeBusHardware(closeBusHardwareId);
        if (resultVoid.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during closeBusHardware: ", resultVoid.getError());
            return;
        }

        ctx.openBusHardwareIds.remove(closeBusHardwareId);

        if (ctx.openBusHardwareIds.isEmpty()) {
            ctx.scannedDeviceIds.clear();
            ctx.activeDevice = new DeviceHandle();
        }

        ctx.openableBusHardwareIds = Menu.getOpenableBusHwIds(ctx);
    }

    /**
    * @brief Close all open bus hardware
    *
    * @param  ctx  the menu context
    * @return      void
    */
    public static void closeAllBusHardware(Context ctx) {
        ctx.waitForUserConfirmation = false;
        StringBuilder errorMessages = new StringBuilder();

        if (ctx.openBusHardwareIds.isEmpty()) {
            MenuUtils.handleErrorMessage(ctx, "No open bus hardware found.", "");
            return;
        }

        for (BusHardwareId openBusHardwareId : ctx.openBusHardwareIds) {
            ctx.connectedDeviceHandles.removeIf(e -> openBusHardwareId.equals(ctx.nanolibAccessor.getDeviceId(e).getResult().getBusHardwareId()));
            ctx.connectableDeviceIds.removeIf(e -> openBusHardwareId.equals(e.getBusHardwareId()));

            ResultVoid resultVoid = ctx.nanolibAccessor.closeBusHardware(openBusHardwareId);
            if (resultVoid.hasError()) {
                errorMessages.append(MenuUtils.handleErrorMessage(ctx, "Error during closeBusHardware: ", resultVoid.getError())).append("\n");
            }
        }

        ctx.errorText = errorMessages.toString();
        ctx.openBusHardwareIds.clear();
        ctx.scannedDeviceIds.clear();
        ctx.openableBusHardwareIds = Menu.getOpenableBusHwIds(ctx);
        ctx.activeDevice = new DeviceHandle();
    }
}

