/**
* Nanotec Nanolib example
* Copyright (C) Nanotec GmbH & Co. KG - All Rights Reserved
*
* This product includes software developed by the
* Nanotec GmbH & Co. KG (http://www.nanotec.com/).
*
* The Nanolib interface headers and the examples source code provided are 
* licensed under the Creative Commons Attribution 4.0 Internaltional License. 
* To view a copy of this license, 
* visit https://creativecommons.org/licenses/by/4.0/ or send a letter to 
* Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
*
* The parts of the library provided in binary format are licensed under 
* the Creative Commons Attribution-NoDerivatives 4.0 International License. 
* To view a copy of this license, 
* visit http://creativecommons.org/licenses/by-nd/4.0/ or send a letter to 
* Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
*
* @file   OdInterfaceFunctionsExample.java
*
* @brief  Definition of object dictionary interface specific functions
*
* @date   29-10-2024
*
* @author Michael Milbradt
*
*/
package com.nanotec.example.NanolibExample;

import java.io.IOException;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.Scanner;

import com.nanotec.nanolib.ObjectDictionary;
import com.nanotec.nanolib.ObjectEntry;
import com.nanotec.nanolib.ObjectSubEntry;
import com.nanotec.nanolib.OdTypesHelper;
import com.nanotec.nanolib.ResultArrayInt;
import com.nanotec.nanolib.ResultInt;
import com.nanotec.nanolib.ResultObjectDictionary;
import com.nanotec.nanolib.ResultString;
import com.nanotec.nanolib.ResultVoid;
import com.nanotec.example.NanolibExample.MenuUtils.Context;
import com.nanotec.example.NanolibExample.MenuUtils.NlcConstants;
import com.nanotec.nanolib.DeviceHandle;
import com.nanotec.nanolib.IntVector;

/**
 * @brief Container class for od inteface functions
 */
public class OdInterfaceFunctionsExample {
    
    /**
     * @brief Function to read a number (no interpretation of the data possible)
     * @param ctx - menu context
     */
    public static void readNumber(Context ctx) {
        ctx.waitForUserConfirmation = true;
    
        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }
    
        System.out.println("Reading mode of operation (" + NlcConstants.OD_MODE_OF_OPERATION.toString() + ") ...");
        ResultInt resultInt = ctx.getNanolibAccessor().readNumber(ctx.activeDevice, NlcConstants.OD_MODE_OF_OPERATION);
        if (resultInt.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during readNumber: ", resultInt.getError());
        }
        System.out.println(NlcConstants.OD_MODE_OF_OPERATION.toString() + " = " + resultInt.getResult());
        System.out.println("This is only the raw value. The OD value might be signed or unsigned up to a total length of 4 bytes\n");
    
        System.out.println("Reading SI unit position (" + NlcConstants.OD_SI_UNIT_POSITION.toString() + ") ...");
        resultInt = ctx.getNanolibAccessor().readNumber(ctx.activeDevice, NlcConstants.OD_SI_UNIT_POSITION);
        if (resultInt.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during readNumber: ", resultInt.getError());
            return;
        }
        System.out.println(NlcConstants.OD_SI_UNIT_POSITION.toString() + " = " + resultInt.getResult());
        System.out.println("This is only the raw value. The OD value might be signed or unsigned up to a total length of 4 bytes");
    }

    /**
     * @brief Function to read a string (string might be zero)
     * @param ctx - menu context
     */
    public static void readString(Context ctx) {
        ctx.waitForUserConfirmation = true;
    
        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }
    
        System.out.println("Reading Nanotec home page string (" + NlcConstants.OD_HOME_PAGE.toString() + ") ...");
        ResultString resultString = ctx.getNanolibAccessor().readString(ctx.activeDevice, NlcConstants.OD_HOME_PAGE);
        if (resultString.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during readString: ", resultString.getError());
        }
        System.out.println(NlcConstants.OD_HOME_PAGE.toString() + " = '" + resultString.getResult() + "'");
    }

    /**
     * @brief Function to read an array (no interpretation of the data possible)
     * @param ctx - menu context
     */
    public static void readArray(Context ctx) {
        ctx.waitForUserConfirmation = true;
    
        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }
    
        System.out.println("Reading device error stack (0x1003) ...");
        ResultArrayInt resultArrayInt = ctx.getNanolibAccessor().readNumberArray(ctx.activeDevice, NlcConstants.OD_ERROR_STACK_INDEX);
        if (resultArrayInt.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during readArray: ", resultArrayInt.getError());
        }
    
        // output only the first field (error count)
        // fields with index > 0 would contain specific stored errors
        IntVector errorStack = resultArrayInt.getResult();
        System.out.println("The error stack has " + errorStack.size() + " elements");
        System.out.println("The first element (error count) is: " + errorStack.get(0));
    }

    /**
     * @brief Function to write a number with certain length
     * @param ctx - menu context
     */
    public static void writeNumber(Context ctx) {
        ctx.waitForUserConfirmation = true;
    
        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }
    
        System.out.println("Writing motor stop command to control word (" + NlcConstants.OD_CONTROL_WORD.toString() + " = 0x06) ...");
        ResultVoid resultVoid = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 6, NlcConstants.OD_CONTROL_WORD, 16);
        if (resultVoid.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during writeNumber: ", resultVoid.getError());
        }
    }

    /**
     * @brief Assign a valid object dictionary to the current active device
     * @param ctx - menu context
     */
    public static void assignObjectDictionary(Context ctx) {
        ctx.waitForUserConfirmation = false;

        String inputPath;

        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        System.out.print("Please enter the directory (path) where the od.xml is located: ");

        // Reading user input
        inputPath = readUserInput();
        
        if (inputPath == null) {
            MenuUtils.handleErrorMessage(ctx, "Error during assignObjectDictionary:", "Invalid path");
            return; // Handle input error
        }

        ResultObjectDictionary resultObjectDictionary = ctx.nanolibAccessor.autoAssignObjectDictionary(
                ctx.activeDevice, inputPath);

        if (resultObjectDictionary.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during assignObjectDictionary: ", resultObjectDictionary.getError());
            return;
        }
    }

    /**
     * @brief Function to read a number (with interpretation of the data)
     * @param ctx - menu context
     */
    public static void readNumberViaDictionaryInterface(Context ctx) {
        ctx.waitForUserConfirmation = true;
    
        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }
    
        ResultObjectDictionary resultObjectDictionary = ctx.getNanolibAccessor().getAssignedObjectDictionary(ctx.activeDevice);
        if (resultObjectDictionary.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during readNumberViaDictionaryInterface: ", resultObjectDictionary.getError());
            return;
        }
    
        if (resultObjectDictionary.getResult().getXmlFileName().getResult().isEmpty()) {
            System.out.println(ctx.lightYellow.getEscapeCode() + "No valid object dictionary assigned. Using fallback method!" + ctx.def.getEscapeCode());
        }
    
        ObjectDictionary objectDictionary = resultObjectDictionary.getResult();
    
        if (!objectDictionary.getDeviceHandle().getResult().equals(ctx.activeDevice)) {
            MenuUtils.handleErrorMessage(ctx, "", "Object dictionary mismatch in readNumberViaDictionaryInterface.");
            return;
        }
    
        System.out.println("Reading mode of operation (" + NlcConstants.OD_MODE_OF_OPERATION.toString() + ") ...");
        ResultInt resultInt = objectDictionary.readNumber(NlcConstants.OD_MODE_OF_OPERATION);
        if (resultInt.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during readNumberViaDictionaryInterface: ", resultInt.getError());
        }
        // OD 0x6060:00 is of type byte so the user has to cast accordingly
        byte modeOfOperation = (byte) resultInt.getResult();
        System.out.println(NlcConstants.OD_MODE_OF_OPERATION.toString() + " = " + modeOfOperation);
    
        System.out.println("Some object entry properties: ");
        ObjectEntry objectEntry = objectDictionary.getObjectEntry(NlcConstants.OD_MODE_OF_OPERATION.getIndex()).getResult();
        System.out.print("Object(" + NlcConstants.OD_MODE_OF_OPERATION.toString() + ").ObjectCode = ");
        String objectCodeString = objectEntry.getObjectCode().toString();
        System.out.println(objectCodeString);
    
        System.out.println("Object(0x6060).DataType = " + OdTypesHelper.objectEntryDataTypeToString(objectEntry.getDataType()));
    
        System.out.println("Some ObjectSubEntry properties: ");
        ObjectSubEntry objectSubEntry = objectDictionary.getObject(NlcConstants.OD_MODE_OF_OPERATION).getResult();
        System.out.println("OdIndex(" + NlcConstants.OD_MODE_OF_OPERATION.toString() + ").DataType = " +
                OdTypesHelper.objectEntryDataTypeToString(objectSubEntry.getDataType()));
        System.out.println("OdIndex(" + NlcConstants.OD_MODE_OF_OPERATION.toString() + ").BitLength = " +
                objectSubEntry.getBitLength() + "\n");
    
        System.out.println("Reading SI unit position (" + NlcConstants.OD_SI_UNIT_POSITION.toString() + ") ...");
        resultInt = objectDictionary.readNumber(NlcConstants.OD_SI_UNIT_POSITION);
        if (resultInt.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during readNumberViaDictionaryInterface: ", resultInt.getError());
            return;
        }
        // OD 0x60A8:00 is of type uint so the user has to cast accordingly
        long unitPosition = resultInt.getResult();
        System.out.println(NlcConstants.OD_SI_UNIT_POSITION.toString() + " = " + unitPosition);
    
        System.out.println("Some object entry properties: ");
        ObjectEntry objectEntry2 = objectDictionary.getObjectEntry(NlcConstants.OD_SI_UNIT_POSITION.getIndex()).getResult();
        System.out.print("Object(" + NlcConstants.OD_SI_UNIT_POSITION.toString() + ").ObjectCode = ");
        objectCodeString = objectEntry2.getObjectCode().toString();
        System.out.println(objectCodeString);
    
        System.out.println("Object(0x60A8).DataType = " +
                OdTypesHelper.objectEntryDataTypeToString(objectEntry2.getDataType()));
    
        System.out.println("Some ObjectSubEntry properties: ");
        ObjectSubEntry objectSubEntry2 = objectDictionary.getObject(NlcConstants.OD_SI_UNIT_POSITION).getResult();
        System.out.println("OdIndex(" + NlcConstants.OD_SI_UNIT_POSITION.toString() + ").DataType = " +
                OdTypesHelper.objectEntryDataTypeToString(objectSubEntry2.getDataType()));
        System.out.println("OdIndex(" + NlcConstants.OD_SI_UNIT_POSITION.toString() + ").BitLength = " +
                objectSubEntry2.getBitLength());
    }

    /**
     * @brief Function to write a number (no length has to be defined)
     * @param ctx - menu context
     */
    public static void writeNumberViaDictionaryInterface(Context ctx) {
        ctx.setWaitForUserConfirmation(true);
    
        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }
    
        ResultObjectDictionary resultObjectDictionary = ctx.getNanolibAccessor().getAssignedObjectDictionary(ctx.activeDevice);
        if (resultObjectDictionary.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during writeNumberViaDictionaryInterface: ", resultObjectDictionary.getError());
            return;
        }
    
        if (resultObjectDictionary.getResult().getXmlFileName().getResult().isEmpty()) {
            System.out.println(ctx.lightYellow.getEscapeCode() + "No valid object dictionary assigned. Using fallback method!" + ctx.def.getEscapeCode());
        }
    
        ObjectDictionary objectDictionary = resultObjectDictionary.getResult();
    
        if (!objectDictionary.getDeviceHandle().getResult().equals(ctx.activeDevice)) {
            MenuUtils.handleErrorMessage(ctx, "", "Object dictionary mismatch in writeNumberViaDictionaryInterface.");
            return;
        }
    
        System.out.println("Writing motor stop command to control word (" + NlcConstants.OD_CONTROL_WORD.toString() + ") with value 0x06 ...");
        long value = 6;
        ResultVoid writeResult = objectDictionary.writeNumber(NlcConstants.OD_CONTROL_WORD, value);
    
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during writeNumberViaDictionaryInterface: ", writeResult.getError());
            return;
        }
    }

    /**
    * @brief Helper method to read user input
    *
    * @param  ctx menu context
    * @return void
    */
    private static String readUserInput() {
        String input = "";
        try {
            input = MenuUtils.reader.readLine();
        } catch (IOException e) {
            return null;
        }
        
        return input.isEmpty() ? null : input;
    }
}
