/**
* Nanotec Nanolib example
* Copyright (C) Nanotec GmbH & Co. KG - All Rights Reserved
*
* This product includes software developed by the
* Nanotec GmbH & Co. KG (http://www.nanotec.com/).
*
* The Nanolib interface headers and the examples source code provided are 
* licensed under the Creative Commons Attribution 4.0 Internaltional License. 
* To view a copy of this license, 
* visit https://creativecommons.org/licenses/by/4.0/ or send a letter to 
* Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
*
* The parts of the library provided in binary format are licensed under 
* the Creative Commons Attribution-NoDerivatives 4.0 International License. 
* To view a copy of this license, 
* visit http://creativecommons.org/licenses/by-nd/4.0/ or send a letter to 
* Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
*
* @file   SamplerFunctionsExample.java
*
* @brief  Definition of sampler specific functions
*
* @date   29-10-2024
*
* @author Michael Milbradt
*
*/
package com.nanotec.example.NanolibExample;

import com.nanotec.example.NanolibExample.MenuUtils.Context;
import com.nanotec.nanolib.DeviceHandle;

/**
 * @brief Container class for sampler functions
 */
public class SamplerFunctionsExample {

    /**
     * @brief Execute the sampler example in normal mode without notification callback.
     *
     * @param ctx - menu context
     */
    public static void executeSamplerWithoutNotificationNormalMode(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        System.out.println("In normal mode the number of samples can be configured.");
        System.out.println("In this example the sampler will run for 5 samples.");

        SamplerExample samplerExample = new SamplerExample(ctx, ctx.activeDevice);
        samplerExample.processSamplerWithoutNotificationNormal();

        System.out.println("Finished");
    }

    /**
     * @brief Execute the sampler example in repetitive mode without notification callback.
     *
     * @param ctx - menu context
     */
    public static void executeSamplerWithoutNotificationRepetitiveMode(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        System.out.println("In repetitive mode the sampler runs until stopped.");
        System.out.println("In this example the sampler will run for 4 iterations and then stop.");

        SamplerExample samplerExample = new SamplerExample(ctx, ctx.activeDevice);
        samplerExample.processSamplerWithoutNotificationRepetitive();

        System.out.println("Finished");
    }

    /**
     * @brief Execute the sampler example in continuous mode without notification callback.
     *
     * @param ctx - menu context
     */
    public static void executeSamplerWithoutNotificationContinuousMode(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        System.out.println("In continuous mode the sampler runs until stopped.");
        System.out.println("In this example the sampler will run for 10 samples and then stop.");

        SamplerExample samplerExample = new SamplerExample(ctx, ctx.activeDevice);
        samplerExample.processSamplerWithoutNotificationContinuous();

        System.out.println("Finished");
    }

    /**
     * @brief Execute the sampler example in normal mode with notification callback.
     *
     * @param ctx - menu context
     */
    public static void executeSamplerWithNotificationNormalMode(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        System.out.println("In normal mode the number of samples can be configured.");
        System.out.println("In this example the sampler will run for 5 samples.");

        SamplerExample samplerExample = new SamplerExample(ctx, ctx.activeDevice);
        samplerExample.processSamplerWithNotificationNormal();

        System.out.println("Finished");
    }

    /**
     * @brief Execute the sampler example in repetitive mode with notification callback.
     *
     * @param ctx - menu context
     */
    public static void executeSamplerWithNotificationRepetitiveMode(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        System.out.println("In repetitive mode the sampler runs until stopped.");
        System.out.println("In this example the sampler will run for 4 iterations and then stop.");

        SamplerExample samplerExample = new SamplerExample(ctx, ctx.activeDevice);
        samplerExample.processSamplerWithNotificationRepetitive();

        System.out.println("Finished");
    }

    /**
     * @brief Execute the sampler example in continuous mode with notification callback.
     *
     * @param ctx - menu context
     */
    public static void executeSamplerWithNotificationContinuousMode(Context ctx) {
        ctx.waitForUserConfirmation = true;

        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "No active device set. Select an active device first.", "");
            return;
        }

        System.out.println("In continuous mode the sampler runs until stopped.");
        System.out.println("In this example the sampler will run for 10 samples and then stop.");

        SamplerExample samplerExample = new SamplerExample(ctx, ctx.activeDevice);
        samplerExample.processSamplerWithNotificationContinuous();

        System.out.println("Finished");
    }
}
