/**
* Nanotec Nanolib example
* Copyright (C) Nanotec GmbH & Co. KG - All Rights Reserved
*
* This product includes software developed by the
* Nanotec GmbH & Co. KG (http://www.nanotec.com/).
*
* The Nanolib interface headers and the examples source code provided are 
* licensed under the Creative Commons Attribution 4.0 Internaltional License. 
* To view a copy of this license, 
* visit https://creativecommons.org/licenses/by/4.0/ or send a letter to 
* Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
*
* The parts of the library provided in binary format are licensed under 
* the Creative Commons Attribution-NoDerivatives 4.0 International License. 
* To view a copy of this license, 
* visit http://creativecommons.org/licenses/by-nd/4.0/ or send a letter to 
* Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
*
* @file   Example.java
*
* @brief  Main function, definition of menu structure etc.
*
* @date   29-10-2024
*
* @author Michael Milbradt
*
*/
package com.nanotec.example.NanolibExample;

import java.util.ArrayList;
import java.util.List;
import com.nanotec.example.NanolibExample.MenuUtils.Context;
import com.nanotec.example.NanolibExample.MenuUtils.Menu;
import com.nanotec.example.NanolibExample.MenuUtils.NlcConstants;
import com.nanotec.example.NanolibExample.MenuUtils.Menu.MenuItem;
import com.nanotec.example.NanolibExample.MenuUtils.Menu.Variant;
import com.nanotec.nanolib.LogLevel;

/**
 * @brief Container class for Main
 */
public class Example {
    /**
    * @brief Creates the connectDeviceMenu menu with dynamic entries linked to connectDevice function
    *
    * @param  ctx menu context
    * @return void
    */
    public static void buildConnectDeviceMenu(Context ctx) {
        Menu connectDeviceMenu = new Menu(NlcConstants.DEVICE_CONNECT_MENU, 
                                                new ArrayList<>(), // Placeholder, dynamically filled
                                                DeviceFunctionsExample::connectDevice);
        connectDeviceMenu.menu(ctx);
    }
    
    /**
    * @brief Creates the disconnectFromDeviceMenu menu with dynamic entries linked to disconnectDevice function
    *
    * @param  ctx menu context
    * @return void
    */
    public static void buildDisconnectDeviceMenu(Context ctx) {
        Menu disconnectFromDeviceMenu = new Menu(NlcConstants.DEVICE_DISCONNECT_MENU, 
                                                  new ArrayList<>(), // Placeholder, dynamically filled
                                                  DeviceFunctionsExample::disconnectDevice);
        disconnectFromDeviceMenu.menu(ctx);
    }
    
    /**
    * @brief Creates the openBusHwMenu menu with dynamic entries linked to openBusHardware function
    *
    * @param  ctx menu context
    * @return void
    */
    public static void buildOpenBusHwMenu(Context ctx) {
        Menu openBusHwMenu = new Menu(NlcConstants.BUS_HARDWARE_OPEN_MI, 
                                       new ArrayList<>(), // Placeholder, dynamically filled
                                       BusFunctionsExample::openBusHardware);
        openBusHwMenu.menu(ctx);
    }
    
    /**
    * @brief Creates the closeBusHwMenu menu with dynamic entries linked to closeBusHardware function
    *
    * @param  ctx menu context
    * @return void
    */
    public static void buildCloseBusHwMenu(Context ctx) {
        Menu closeBusHwMenu = new Menu(NlcConstants.BUS_HARDWARE_CLOSE_MI, 
                                        new ArrayList<>(), // Placeholder, dynamically filled
                                        BusFunctionsExample::closeBusHardware);
        closeBusHwMenu.menu(ctx);
    }
    
    /**
    * @brief Creates the buildSelectActiveDeviceMenu menu with dynamic entries linked to selectActiveDevice function
    *
    * @param  ctx menu context
    * @return void
    */
    public static void buildSelectActiveDeviceMenu(Context ctx) {
        Menu selectActiveDeviceMenu = new Menu(NlcConstants.DEVICE_SELECT_ACTIVE_MENU, 
                                               new ArrayList<>(), // Placeholder, dynamically filled
                                               DeviceFunctionsExample::selectActiveDevice);
        selectActiveDeviceMenu.menu(ctx);
    }
    
    /**
    * @brief Main function
    *
    * @param  args input arguments
    * @return void
    */
    public static void main(String[] args) {

        Context context = new Context(); // the menu context
        LoggingCallbackExample loggingCallback = new LoggingCallbackExample();
        ScanBusCallbackExample scanBusCallback = new ScanBusCallbackExample();
        DataTransferCallbackExample dataTransferCallback = new DataTransferCallbackExample();

        // Setup menu context
        context.setCurrentLogLevel(LogLevel.Off); // no logging output at start
        context.setLoggingCallbackActive(false); // no logging callback active
        context.setLoggingCallback(loggingCallback); // store pointer to logging callback object
        context.setScanBusCallback(scanBusCallback); // store pointer to scan bus callback object
        context.setDataTransferCallback(dataTransferCallback); // store pointer to data transfer callback object
        context.setWaitForUserConfirmation(false); // flag to stop at end of a menu function
        // Set log level to off
        context.getNanolibAccessor().setLoggingLevel(LogLevel.Off);

        // build the motorMenu menu as a submenu
	// every menu has a title and menu items
	// every menu item has a name and a functional interface to a function or a menu entry to another menu
	// on dynamic menus the function is used to build the underlaying menu
        Menu motorMenu = new Menu(NlcConstants.MOTOR_EXAMPLE_MENU,
                List.of(
                        new MenuItem(NlcConstants.MOTOR_AUTO_SETUP_MI, (Variant)MotorFunctionsExample::motorAutoSetup),
                        new MenuItem(NlcConstants.MOTOR_VELOCITY_MI, (Variant)MotorFunctionsExample::executeProfileVelocityMode),
                        new MenuItem(NlcConstants.MOTOR_POSITIONING_MI, (Variant)MotorFunctionsExample::executePositioningMode)
                ), null);

        // build the samplerMenu menu as a submenu
        Menu samplerMenu = new Menu(NlcConstants.SAMPLER_EXAMPLE_MENU,
                List.of(
                        new MenuItem(NlcConstants.SAMPLER_NORMAL_WO_NOTIFY_MI, (Variant)SamplerFunctionsExample::executeSamplerWithoutNotificationNormalMode),
                        new MenuItem(NlcConstants.SAMPLER_REPETETIVE_WO_NOTIFY_MI, (Variant)SamplerFunctionsExample::executeSamplerWithoutNotificationRepetitiveMode),
                        new MenuItem(NlcConstants.SAMPLER_CONTINUOUS_WO_NOTIFY_MI, (Variant)SamplerFunctionsExample::executeSamplerWithoutNotificationContinuousMode),
                        new MenuItem(NlcConstants.SAMPLER_NORMAL_WITH_NOTIFY_MI, (Variant)SamplerFunctionsExample::executeSamplerWithNotificationNormalMode),
                        new MenuItem(NlcConstants.SAMPLER_REPETETIVE_WITH_NOTIFY_MI, (Variant)SamplerFunctionsExample::executeSamplerWithNotificationRepetitiveMode),
                        new MenuItem(NlcConstants.SAMPLER_CONTINUOUS_WITH_NOTIFY_MI, (Variant)SamplerFunctionsExample::executeSamplerWithNotificationContinuousMode)
                ), null);

        // build the logCallbackMenu menu as a submenu
        Menu logCallbackMenu = new Menu(NlcConstants.LOG_CALLBACK_MENU,
                List.of(
                        new MenuItem(NlcConstants.LOG_CALLBACK_CORE_MI, (Variant)LoggingFunctionsExample::setLoggingCallback),
                        new MenuItem(NlcConstants.LOG_CALLBACK_CANOPEN_MI, (Variant)LoggingFunctionsExample::setLoggingCallback),
                        new MenuItem(NlcConstants.LOG_CALLBACK_ETHERCAT_MI, (Variant)LoggingFunctionsExample::setLoggingCallback),
                        new MenuItem(NlcConstants.LOG_CALLBACK_MODBUS_MI, (Variant)LoggingFunctionsExample::setLoggingCallback),
                        new MenuItem(NlcConstants.LOG_CALLBACK_REST_MI, (Variant)LoggingFunctionsExample::setLoggingCallback),
                        new MenuItem(NlcConstants.LOG_CALLBACK_USB_MI, (Variant)LoggingFunctionsExample::setLoggingCallback),
                        new MenuItem(NlcConstants.LOG_CALLBACK_DEACTIVATE_MI, (Variant)LoggingFunctionsExample::setLoggingCallback)
                ), null);

        // build the logLevelgMenu menu as a submenu
        Menu logLevelMenu = new Menu(NlcConstants.LOG_LEVEL_MENU,
                List.of(
                        new MenuItem(NlcConstants.LOG_LEVEL_TRACE_MI, (Variant)LoggingFunctionsExample::setLogLevel),
                        new MenuItem(NlcConstants.LOG_LEVEL_DEBUG_MI, (Variant)LoggingFunctionsExample::setLogLevel),
                        new MenuItem(NlcConstants.LOG_LEVEL_INFO_MI, (Variant)LoggingFunctionsExample::setLogLevel),
                        new MenuItem(NlcConstants.LOG_LEVEL_WARN_MI, (Variant)LoggingFunctionsExample::setLogLevel),
                        new MenuItem(NlcConstants.LOG_LEVEL_ERROR_MI, (Variant)LoggingFunctionsExample::setLogLevel),
                        new MenuItem(NlcConstants.LOG_LEVEL_CRITICAL_MI, (Variant)LoggingFunctionsExample::setLogLevel),
                        new MenuItem(NlcConstants.LOG_LEVEL_OFF_MI, (Variant)LoggingFunctionsExample::setLogLevel)
                ), null);

        // build the loggingMenu menu as a submenu
        Menu loggingMenu = new Menu(NlcConstants.LOGGING_MENU,
                List.of(
                        new MenuItem(NlcConstants.LOGGING_SET_LOG_LEVEL_MI, null, logLevelMenu),
                        new MenuItem(NlcConstants.LOGGING_SET_LOG_CALLBACK_MI, null, logCallbackMenu)
                ), null);

        // Build the odAccessMenu as a submenu
        Menu odAccessMenu = new Menu(NlcConstants.OD_INTERFACE_MENU,
                List.of(
                        new MenuItem(NlcConstants.OD_ASSIGN_OD_MI, (Variant)OdInterfaceFunctionsExample::assignObjectDictionary),
                        new MenuItem(NlcConstants.OD_READ_NUMBER_MI, (Variant)OdInterfaceFunctionsExample::readNumber),
                        new MenuItem(NlcConstants.OD_READ_NUMBER_VIA_OD_MI, (Variant)OdInterfaceFunctionsExample::readNumberViaDictionaryInterface),
                        new MenuItem(NlcConstants.OD_WRITE_NUMBER_MI, (Variant)OdInterfaceFunctionsExample::writeNumber),
                        new MenuItem(NlcConstants.OD_WRITE_NUMBER_VIA_OD_MI, (Variant)OdInterfaceFunctionsExample::writeNumberViaDictionaryInterface),
                        new MenuItem(NlcConstants.OD_READ_STRING_MI, (Variant)OdInterfaceFunctionsExample::readString),
                        new MenuItem(NlcConstants.OD_READ_BYTES_MI, (Variant)OdInterfaceFunctionsExample::readArray)
                ), null);

        // build the deviceInfoMenu menu as a submenu
        Menu deviceInfoMenu = new Menu(NlcConstants.DEVICE_INFORMATION_MENU,
                List.of(
                        new MenuItem(NlcConstants.DEVICE_GET_VENDOR_ID_MI, (Variant)DeviceFunctionsExample::getDeviceVendorId),
                        new MenuItem(NlcConstants.DEVICE_GET_PRODUCT_CODE_MI, (Variant)DeviceFunctionsExample::getDeviceProductCode),
                        new MenuItem(NlcConstants.DEVICE_GET_DEVICE_NAME_MI, (Variant)DeviceFunctionsExample::getDeviceName),
                        new MenuItem(NlcConstants.DEVICE_GET_HW_VERSION_MI, (Variant)DeviceFunctionsExample::getDeviceHardwareVersion),
                        new MenuItem(NlcConstants.DEVICE_GET_FW_BUILD_ID_MI, (Variant)DeviceFunctionsExample::getDeviceFirmwareBuildId),
                        new MenuItem(NlcConstants.DEVICE_GET_BL_BUILD_ID_MI, (Variant)DeviceFunctionsExample::getDeviceBootloaderBuildId),
                        new MenuItem(NlcConstants.DEVICE_GET_SERIAL_NUMBER_MI, (Variant)DeviceFunctionsExample::getDeviceSerialNumber),
                        new MenuItem(NlcConstants.DEVICE_GET_UNIQUE_ID_MI, (Variant)DeviceFunctionsExample::getDeviceUid),
                        new MenuItem(NlcConstants.DEVICE_GET_BL_VERSION_MI, (Variant)DeviceFunctionsExample::getDeviceBootloaderVersion),
                        new MenuItem(NlcConstants.DEVICE_GET_HW_GROUP_MI, (Variant)DeviceFunctionsExample::getDeviceHardwareGroup),
                        new MenuItem(NlcConstants.DEVICE_GET_CON_STATE_MI, (Variant)DeviceFunctionsExample::getConnectionState)
                ), null);

        // build the deviceMenu menu as a submenu
        Menu deviceMenu = new Menu(NlcConstants.DEVICE_MENU,
                List.of(
                        new MenuItem(NlcConstants.DEVICE_SCAN_MI, (Variant)DeviceFunctionsExample::scanDevices),
                        new MenuItem(NlcConstants.DEVICE_CONNECT_MENU, (Variant)Example::buildConnectDeviceMenu),
                        new MenuItem(NlcConstants.DEVICE_DISCONNECT_MENU, (Variant)Example::buildDisconnectDeviceMenu),
                        new MenuItem(NlcConstants.DEVICE_SELECT_ACTIVE_MENU, (Variant)Example::buildSelectActiveDeviceMenu),
                        new MenuItem(NlcConstants.DEVICE_REBOOT_MI, (Variant)DeviceFunctionsExample::rebootDevice),
                        new MenuItem(NlcConstants.DEVICE_INFORMATION_MENU, null, deviceInfoMenu), // Direct reference to another menu
                        new MenuItem(NlcConstants.DEVICE_UPDATE_FW_MI, (Variant)DeviceFunctionsExample::updateFirmware),
                        new MenuItem(NlcConstants.DEVICE_UPDATE_BL_MI, (Variant)DeviceFunctionsExample::updateBootloader),
                        new MenuItem(NlcConstants.DEVICE_UPLOAD_NANOJ_MI, (Variant)DeviceFunctionsExample::uploadNanoJ),
                        new MenuItem(NlcConstants.DEVICE_RUN_NANOJ_MI, (Variant)DeviceFunctionsExample::runNanoJ),
                        new MenuItem(NlcConstants.DEVICE_STOP_NANOJ_MI, (Variant)DeviceFunctionsExample::stopNanoJ),
                        new MenuItem(NlcConstants.DEVICE_GET_ERROR_FIELD_MI, (Variant)DeviceFunctionsExample::getErrorFields),
                        new MenuItem(NlcConstants.DEVICE_RESTORE_ALL_DEFAULT_PARAMS_MI, (Variant)DeviceFunctionsExample::restoreDefaults)
                ), null);

        // build the busHwMenu menu as a submenu
        Menu busHwMenu = new Menu(NlcConstants.BUS_HARDWARE_MENU, 
                List.of(
                        new MenuItem(NlcConstants.BUS_HARDWARE_SCAN_MI, (Variant)BusFunctionsExample::scanBusHardware),
                        new MenuItem(NlcConstants.BUS_HARDWARE_OPEN_MI, (Variant)Example::buildOpenBusHwMenu),
                        new MenuItem(NlcConstants.BUS_HARDWARE_CLOSE_MI, (Variant)Example::buildCloseBusHwMenu),
                        new MenuItem(NlcConstants.BUS_HARDWARE_CLOSE_ALL_MI, (Variant)BusFunctionsExample::closeAllBusHardware)        
                        ), null);

        // build the mainMenu menu as a mainmenu
        Menu mainMenu = new Menu(NlcConstants.MAIN_MENU,
                List.of(
                        new MenuItem(NlcConstants.BUS_HARDWARE_MENU, null, busHwMenu),
                        new MenuItem(NlcConstants.DEVICE_MENU, null, deviceMenu),
                        new MenuItem(NlcConstants.OD_INTERFACE_MENU, null, odAccessMenu),
                        new MenuItem(NlcConstants.LOGGING_MENU, null, loggingMenu),
                        new MenuItem(NlcConstants.SAMPLER_EXAMPLE_MENU, null, samplerMenu),
                        new MenuItem(NlcConstants.MOTOR_EXAMPLE_MENU, null, motorMenu),
                        new MenuItem(NlcConstants.PROFINET_EXAMPLE_MI, (Variant)ProfinetFunctionsExample::profinetDCPExample)
                ), null);

        // Start the main menu
        mainMenu.menu(context);

        // Close all opened bus hardware
        BusFunctionsExample.closeAllBusHardware(context);
    }
}
