/**
* Nanotec Nanolib example
* Copyright (C) Nanotec GmbH & Co. KG - All Rights Reserved
*
* This product includes software developed by the
* Nanotec GmbH & Co. KG (http://www.nanotec.com/).
*
* The Nanolib interface headers and the examples source code provided are 
* licensed under the Creative Commons Attribution 4.0 Internaltional License. 
* To view a copy of this license, 
* visit https://creativecommons.org/licenses/by/4.0/ or send a letter to 
* Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
*
* The parts of the library provided in binary format are licensed under 
* the Creative Commons Attribution-NoDerivatives 4.0 International License. 
* To view a copy of this license, 
* visit http://creativecommons.org/licenses/by-nd/4.0/ or send a letter to 
* Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
*
* @file   MotorFunctionsExample.java
*
* @brief  TODO Definition of motor specific functions
*
* @date   29-10-2024
*
* @author Michael Milbradt
*
*/
package com.nanotec.example.NanolibExample;

import com.nanotec.example.NanolibExample.MenuUtils.Context;
import com.nanotec.example.NanolibExample.MenuUtils.NlcConstants;
import com.nanotec.nanolib.DeviceHandle;
import com.nanotec.nanolib.ResultVoid;
import com.nanotec.nanolib.ResultInt;;

/**
 * @brief Container class for motor functions
 */
public class MotorFunctionsExample {

    /**
     * @brief Determine motor parameters and store them on the device
     * @param ctx - menu context
     */
    public static void motorAutoSetup(Context ctx) {
        ctx.setWaitForUserConfirmation(true);
    
        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "", "No active device set. Select an active device first.");
            return;
        }
    
        System.out.println();
        System.out.println(ctx.lightYellow.getEscapeCode());
        System.out.println("Please note the following requirements for performing the auto-setup:");
        System.out.println("- The motor must be unloaded.");
        System.out.println("- The motor must not be touched.");
        System.out.println("- The motor must be able to rotate freely in any direction.");
        System.out.println("- No NanoJ program may be running." + ctx.def.getEscapeCode());
    
        System.out.print("Do you want to continue? ");
        Character result = MenuUtils.getchr("Do you want to continue?", "y/n", 'y');
        if (!result.equals('y')) {
            ctx.setWaitForUserConfirmation(false);
            return;
        }
    
        // Stop a possibly running NanoJ program
        ResultVoid writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x00, NlcConstants.OD_NANOJ_CONTROL, 32);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during motorAutoSetup: ", writeResult.getError());
            return;
        }
    
        // Switch the state machine to "voltage enabled"
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x06, NlcConstants.OD_CONTROL_WORD, 16);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during motorAutoSetup: ", writeResult.getError());
            return;
        }
    
        // Set mode of operation to auto-setup
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0xFE, NlcConstants.OD_MODE_OF_OPERATION, 8);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during motorAutoSetup: ", writeResult.getError());
            return;
        }
    
        // Switch on
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x07, NlcConstants.OD_CONTROL_WORD, 16);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during motorAutoSetup: ", writeResult.getError());
            return;
        }
    
        // Switch the state machine to "enable operation"
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x0F, NlcConstants.OD_CONTROL_WORD, 16);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during motorAutoSetup: ", writeResult.getError());
            return;
        }
    
        // Run auto setup
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x1F, NlcConstants.OD_CONTROL_WORD, 16);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during motorAutoSetup: ", writeResult.getError());
            return;
        }
    
        System.out.println("Motor auto setup is running, please wait ...");
    
        // Wait until auto setup is finished, check status word
        while (true) {
            ResultInt readNumberResult = ctx.getNanolibAccessor().readNumber(ctx.activeDevice, NlcConstants.OD_STATUS_WORD);
            if (readNumberResult.hasError()) {
                MenuUtils.handleErrorMessage(ctx, "Error during motorAutoSetup: ", readNumberResult.getError());
                return;
            }
    
            // Finish if bits 12, 9, 5, 4, 2, 1, 0 are set
            if ((readNumberResult.getResult() & 0x1237) == 0x1237) {
                break;
            }
        }
    
        // Reboot current active device
        System.out.println("Rebooting ...");
        ResultVoid rebootResult = ctx.getNanolibAccessor().rebootDevice(ctx.activeDevice);
        if (rebootResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during motorAutoSetup: ", rebootResult.getError());
            return;
        }
        System.out.println("Motor auto setup finished.");
    }

    /**
     * @brief Function to demonstrate how to move a motor in profile velocity mode
     * @param ctx - menu context
     */
    public static void executeProfileVelocityMode(Context ctx) {
        ctx.setWaitForUserConfirmation(true);
    
        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "", "No active device set. Select an active device first.");
            return;
        }
    
        System.out.println("This example lets the motor run in Profile Velocity mode ...");
    
        // Stop a possibly running NanoJ program
        ResultVoid writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x00, NlcConstants.OD_NANOJ_CONTROL, 32);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executeProfileVelocityMode: ", writeResult.getError());
            return;
        }
    
        // Choose Profile Velocity mode
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x03, NlcConstants.OD_MODE_OF_OPERATION, 8);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executeProfileVelocityMode: ", writeResult.getError());
            return;
        }
    
        // Set the desired speed in rpm (60)
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x3C, NlcConstants.OD_TARGET_VELOCITY, 32);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executeProfileVelocityMode: ", writeResult.getError());
            return;
        }
    
        // Switch the state machine to "operation enabled"
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x06, NlcConstants.OD_CONTROL_WORD, 16);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executeProfileVelocityMode: ", writeResult.getError());
            return;
        }
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x07, NlcConstants.OD_CONTROL_WORD, 16);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executeProfileVelocityMode: ", writeResult.getError());
            return;
        }
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x0F, NlcConstants.OD_CONTROL_WORD, 16);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executeProfileVelocityMode: ", writeResult.getError());
            return;
        }
        System.out.println("Motor is running clockwise ...");
    
        // Let the motor run for 3s
        try {
            Thread.sleep(3000);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    
        // Stop the motor
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x06, NlcConstants.OD_CONTROL_WORD, 16);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executeProfileVelocityMode: ", writeResult.getError());
            return;
        }
    
        // Set the desired speed in rpm (60), now counterclockwise
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, -0x3C, NlcConstants.OD_TARGET_VELOCITY, 32);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executeProfileVelocityMode: ", writeResult.getError());
            return;
        }
    
        // Start the motor
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x0F, NlcConstants.OD_CONTROL_WORD, 16);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executeProfileVelocityMode: ", writeResult.getError());
            return;
        }
        System.out.println("Motor is running counterclockwise ...");
    
        // Let the motor run for 3s
        try {
            Thread.sleep(3000);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    
        // Stop the motor
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x06, NlcConstants.OD_CONTROL_WORD, 16);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executeProfileVelocityMode: ", writeResult.getError());
            return;
        }
    }

    /**
     * @brief Function to demonstrate how to move a motor in positioning mode
     * @param ctx - menu context
     */
    public static void executePositioningMode(Context ctx) {
        ctx.setWaitForUserConfirmation(true);
    
        if (ctx.activeDevice.equals(new DeviceHandle())) {
            MenuUtils.handleErrorMessage(ctx, "", "No active device set. Select an active device first.");
            return;
        }
    
        System.out.println("This example lets the motor run in Profile Position mode ...");
    
        // Stop a possibly running NanoJ program
        ResultVoid writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x00, NlcConstants.OD_NANOJ_CONTROL, 32);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executePositioningMode: ", writeResult.getError());
            return;
        }
    
        // Choose Profile Position mode
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x01, NlcConstants.OD_MODE_OF_OPERATION, 8);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executePositioningMode: ", writeResult.getError());
            return;
        }
    
        // Set the desired speed in rpm (60)
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x3C, NlcConstants.OD_PROFILE_VELOCITY, 32);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executePositioningMode: ", writeResult.getError());
            return;
        }
    
        // Set the desired target position (36000)
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x8CA0, NlcConstants.OD_TARGET_POSITION, 32);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executePositioningMode: ", writeResult.getError());
            return;
        }
    
        // Switch the state machine to "operation enabled"
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x06, NlcConstants.OD_CONTROL_WORD, 16);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executePositioningMode: ", writeResult.getError());
            return;
        }
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x07, NlcConstants.OD_CONTROL_WORD, 16);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executePositioningMode: ", writeResult.getError());
            return;
        }
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x0F, NlcConstants.OD_CONTROL_WORD, 16);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executePositioningMode: ", writeResult.getError());
            return;
        }
    
        // Move the motor to the desired target position relatively
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x5F, NlcConstants.OD_CONTROL_WORD, 16);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executePositioningMode: ", writeResult.getError());
            return;
        }
        System.out.println("Motor is running clockwise until position is reached ...");
    
        while (true) {
            ResultInt readResult = ctx.getNanolibAccessor().readNumber(ctx.activeDevice, NlcConstants.OD_STATUS_WORD);
            if (readResult.hasError()) {
                MenuUtils.handleErrorMessage(ctx, "Error during executePositioningMode: ", readResult.getError());
                // Try to stop motor
                ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x06, NlcConstants.OD_CONTROL_WORD, 16);
                break;
            }
    
            if ((readResult.getResult() & 0x1400) == 0x1400) {
                break;
            }
        }
    
        // Stop the motor
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x06, NlcConstants.OD_CONTROL_WORD, 16);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executePositioningMode: ", writeResult.getError());
            return;
        }
    
        // Set the desired target position (-36000)
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, -0x8CA0, NlcConstants.OD_TARGET_POSITION, 32);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executePositioningMode: ", writeResult.getError());
            return;
        }
    
        // State machine operation enabled
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x0F, NlcConstants.OD_CONTROL_WORD, 16);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executePositioningMode: ", writeResult.getError());
            return;
        }
    
        // Move the motor to the desired target position relatively
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x5F, NlcConstants.OD_CONTROL_WORD, 16);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executePositioningMode: ", writeResult.getError());
            return;
        }
        System.out.println("Motor is running counterclockwise until position is reached ...");
    
        while (true) {
            ResultInt readResult = ctx.getNanolibAccessor().readNumber(ctx.activeDevice, NlcConstants.OD_STATUS_WORD);
            if (readResult.hasError()) {
                MenuUtils.handleErrorMessage(ctx, "Error during executePositioningMode: ", readResult.getError());
                // Try to stop motor
                ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x06, NlcConstants.OD_CONTROL_WORD, 16);
                break;
            }
    
            if ((readResult.getResult() & 0x1400) == 0x1400) {
                break;
            }
        }
    
        // Stop the motor
        writeResult = ctx.getNanolibAccessor().writeNumber(ctx.activeDevice, 0x06, NlcConstants.OD_CONTROL_WORD, 16);
        if (writeResult.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during executePositioningMode: ", writeResult.getError());
            return;
        }
    }
}
