/**
* Nanotec Nanolib example
* Copyright (C) Nanotec GmbH & Co. KG - All Rights Reserved
*
* This product includes software developed by the
* Nanotec GmbH & Co. KG (http://www.nanotec.com/).
*
* The Nanolib interface headers and the examples source code provided are 
* licensed under the Creative Commons Attribution 4.0 Internaltional License. 
* To view a copy of this license, 
* visit https://creativecommons.org/licenses/by/4.0/ or send a letter to 
* Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
*
* The parts of the library provided in binary format are licensed under 
* the Creative Commons Attribution-NoDerivatives 4.0 International License. 
* To view a copy of this license, 
* visit http://creativecommons.org/licenses/by-nd/4.0/ or send a letter to 
* Creative Commons, PO Box 1866, Mountain View, CA 94042, USA.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
*
* @file   SamplerExample.java
*
* @brief  Definition of sampler example class
*
* @date   29-10-2024
*
* @author Michael Milbradt
*
*/
package com.nanotec.example.NanolibExample;

import java.util.List;

import com.nanotec.nanolib.OdIndex;
import com.nanotec.nanolib.OdIndexVector;
import com.nanotec.nanolib.ResultVoid;
import com.nanotec.nanolib.SampleData;
import com.nanotec.nanolib.SampleDataVector;
import com.nanotec.nanolib.SampledValue;
import com.nanotec.nanolib.SampledValueVector;
import com.nanotec.nanolib.SamplerTriggerCondition;
import com.nanotec.nanolib.SamplerConfiguration;
import com.nanotec.nanolib.SamplerMode;
import com.nanotec.nanolib.SamplerState;
import com.nanotec.nanolib.SamplerNotify;
import com.nanotec.nanolib.SamplerTrigger;
import com.nanotec.example.NanolibExample.MenuUtils.Context;
import com.nanotec.nanolib.DeviceHandle;

/**
 * @brief Sampler example class
 */
public class SamplerExample {

	/**
	 * @brief Inner class TrackedAddress
	 */
    public static class TrackedAddress {
        private String nameString;
        private OdIndex odIndex;

        /**
		 * @brief Constructor with parameters
		 * 
         * @param nameString the name of the od to track
         * @param odIndex the object index of the od to track
         */
        public TrackedAddress(String nameString, OdIndex odIndex) {
            this.nameString = nameString;
            this.odIndex = odIndex;
        }

		/**
		 * @brief Get the name
		 * 
         * @returns name as String
         */
        public String getNameString() {
            return nameString;
        }

		/**
		 * @brief Set the name of the od to track
		 * 
         * @param nameString the name of the od to track
         */
        public void setNameString(String nameString) {
            this.nameString = nameString;
        }

		/**
		 * @brief Get the od index
		 * 
         * @return index of od as OdIndex
         */
        public OdIndex getOdIndex() {
            return odIndex;
        }

		/**
		 * @brief Set the od index of the od to track
		 * 
         * @param odIndex index of od
         */
        public void setOdIndex(OdIndex odIndex) {
            this.odIndex = odIndex;
        }

    }

    /**
	 * @brief Callback class derived from SamplerNotify.
	 */
	private class SamplerNotifyExample extends SamplerNotify // override
	{
		private boolean samplerRunning = false;
		private SamplerExample samplerExample = null;

		/**
		 * @brief Constructor with parameter
		 * @param example the SamplerExample to notify
		 */
		public SamplerNotifyExample(SamplerExample example) {
			this.samplerExample = example;
			this.samplerRunning = true;
		}

		/**
		 * @brief Check if sampler is running
		 * 
		 * @return true if sampler is running, false otherwise
		 */
		public boolean isRunning() {
			return this.samplerRunning;
		}

		/**
		 * @brief Set sampler running to false
		 */
		public void setInactive() {
			this.samplerRunning = false;
		}

		// Be aware that notifications are executed in the context of separate threads
		// other than thread that started the sampler.
		//
		// Be careful when calling Nanolib functionality here, as doing so may cause
		// this method
		// to be called recursively, potentially causing your application to deadlock.
		//
		// For the same reason, this method should not throw exceptions.

		/**
		 * @brief Callback used by Sampler.
		 * 
		 * @param lastError Last error occured
		 * @param samplerState State of the sampler
		 * @param sampleDatas Sampled data
		 * @param applicationData not used
		 */
		@Override
		public void notify(ResultVoid lastError, SamplerState samplerState, SampleDataVector sampleDatas,
				long applicationData) {
			if (!samplerRunning)
				throw new RuntimeException("Sampler not running");

			if (!sampleDatas.isEmpty())
				this.samplerExample.processSampledData(sampleDatas);

			if (samplerState == SamplerState.Failed) {
				try {
					this.samplerExample.handleSamplerFailed(lastError);
				} catch (Exception e) {
					// see comment above
				}
			}

			if ((samplerState != SamplerState.Ready) && (samplerState != SamplerState.Running)) {
				// It's now safe to destroy this notification object
				samplerRunning = false;
			}
		}
	}

    // Constants
	// add two tracked objects
    private static final List<TrackedAddress> trackedAddresses = List.of(
        new TrackedAddress("Up time", new OdIndex(0x230F, (short)0x00)),
        new TrackedAddress("Temperature", new OdIndex(0x4014, (short)0x03))
    );
    
    private static final OdIndex triggerAddress = new OdIndex(0x2400, (short)0x01);
    private static final SamplerTriggerCondition triggerCondition = SamplerTriggerCondition.TC_GREATER;
    private static final long triggerValue = 10;
    private static final long triggerValueInactive = triggerValue;
    private static final long triggerValueActive = triggerValue + 1;
    private static final int periodMilliseconds = 1000;

    private Context ctx; // store menu context
    private DeviceHandle deviceHandle; // device handle for current active device
    private long lastIteration = Long.MAX_VALUE;
    private int sampleNumber = 0;
	private boolean headerPrinted = false;

	/**
	 * @brief Construtor with parameters
	 * 
	 * @param menuContext the menu context
	 * @param connectedDeviceHandle device handle for current active device
	 */
    public SamplerExample(Context menuContext, DeviceHandle connectedDeviceHandle) {
        this.ctx = menuContext;
        this.deviceHandle = connectedDeviceHandle;
    }

	/**
	 * @brief Call all functions
	 */
    public void process() {
        processExamplesWithoutNotification();
        processExamplesWithNotification();
    }

	/**
	 * @brief Call functions without using notification callback
	 */
    public void processExamplesWithoutNotification() {
        processSamplerWithoutNotificationNormal();
        processSamplerWithoutNotificationRepetitive();
        processSamplerWithoutNotificationContinuous();
    }

	/**
	 * @brief Configures and starts sampling process in normal mode without notificaion
	 */
	public void processSamplerWithoutNotificationNormal() {
		System.out.println("\nSampler without notification in normal mode:");

		configure(SamplerMode.Normal);

		start(null);

		SamplerState samplerState = SamplerState.Ready;

		do {
			try {
				Thread.sleep(SamplerExample.periodMilliseconds);
			} catch (InterruptedException e) {
				samplerState = SamplerState.Failed;
				break;
			}
			processSampledData(getSamplerData());
			samplerState = getSamplerState();

		} while ((samplerState == SamplerState.Ready)
				|| (samplerState == SamplerState.Running));

		// Process any remaining data
		processSampledData(getSamplerData());

		if (samplerState == SamplerState.Failed)
			handleSamplerFailed(null);
	}

	/**
	 * @brief Configures and starts sampling process in repetitive mode without notificaion
	 */
	public void processSamplerWithoutNotificationRepetitive() {
		System.out.println("\nSampler without notification in repetative mode:");

		configure(SamplerMode.Repetitive);
		try {
			start(null);
		} catch (Exception e) {
			handleSamplerFailed(null);
		}

		SamplerState samplerState = SamplerState.Ready;

		// wait for sampler to start (running)
		do {
			try {
				Thread.sleep(Integer.valueOf(50));
				samplerState = getSamplerState();
			} catch (InterruptedException e) {
				samplerState = SamplerState.Failed;
			}
		} while ((samplerState != SamplerState.Running)
				&& (samplerState != SamplerState.Failed));

		// process sampler data
		do {
			try {
				Thread.sleep(SamplerExample.periodMilliseconds);
			} catch (InterruptedException e) {
				samplerState = SamplerState.Failed;
				break;
			}
			processSampledData(getSamplerData());

			if (this.lastIteration >= 4) {
				try {
					// In repetative mode the sampler will continue to run until it is stopped or an
					// error occurs
					ctx.nanolibAccessor.getSamplerInterface().stop(deviceHandle);
					break;
				} catch (Exception e) {
					handleSamplerFailed(null);
				}
			}
			samplerState = getSamplerState();

		} while ((samplerState == SamplerState.Ready)
				|| (samplerState == SamplerState.Running));

		// Process any remaining data
		processSampledData(getSamplerData());

		if (samplerState == SamplerState.Failed)
			handleSamplerFailed(null);
	}

	/**
	 * @brief Configures and starts sampling process in continuous mode without notificaion
	 */
	public void processSamplerWithoutNotificationContinuous() {
		System.out.println("\nSampler without notification in continuous mode:");

		configure(SamplerMode.Continuous);
		try {
			start(null);
		} catch (Exception e) {
			handleSamplerFailed(null);
		}

		SamplerState samplerState = SamplerState.Ready;
		int maxCycles = 10;
		int cycles = 0;

		do {
			try {
				Thread.sleep(SamplerExample.periodMilliseconds);
			} catch (InterruptedException e) {
				samplerState = SamplerState.Failed;
				break;
			}
			processSampledData(getSamplerData());

			if (++cycles == maxCycles) {
				try {
					// In continuous mode the sampler will continue to run until it is stopped or an
					// error occurs
					ctx.nanolibAccessor.getSamplerInterface().stop(deviceHandle);
					break;
				} catch (Exception e) {
					handleSamplerFailed(null);
				}
			}

			samplerState = getSamplerState();

		} while ((samplerState == SamplerState.Ready)
				|| (samplerState == SamplerState.Running));

		// Process any remaining data
		processSampledData(getSamplerData());

		if (samplerState == SamplerState.Failed)
			handleSamplerFailed(null);
	}

	/**
	 * @brief Call functions using notification callback
	 */
    public void processExamplesWithNotification() {
        processSamplerWithNotificationNormal();
        processSamplerWithNotificationRepetitive();
        processSamplerWithNotificationContinuous();
    }

	/**
	 * @brief Configures and starts sampling process in normal mode with notificaion
	 */
	public void processSamplerWithNotificationNormal() {
		System.out.println("\nSampler with notification in normal mode:");

		configure(SamplerMode.Normal);

		SamplerNotifyExample samplerNotify = new SamplerNotifyExample(this);

		start(samplerNotify);
		while (samplerNotify.isRunning()) {
			try {
				Thread.sleep(periodMilliseconds);
			} catch (InterruptedException e) {
				samplerNotify.setInactive();
			}
		}
	}

	/**
	 * @brief Configures and starts sampling process in repetitive mode with notificaion
	 */
	public void processSamplerWithNotificationRepetitive() {
		System.out.println("\nSampler with notification in repetative mode:");

		configure(SamplerMode.Repetitive);

		SamplerNotifyExample samplerNotify = new SamplerNotifyExample(this);

		start(samplerNotify);
		SamplerState samplerState = SamplerState.Ready;

		// wait for sampler to start (running)
		do {
			try {
				Thread.sleep(Integer.valueOf(50));
				samplerState = getSamplerState();
			} catch (InterruptedException e) {
				samplerState = SamplerState.Failed;
			}
		} while ((samplerState != SamplerState.Running)
				&& (samplerState != SamplerState.Failed));

		// process sampler data
		while (samplerNotify.isRunning()) {
			try {
				Thread.sleep(periodMilliseconds);
			} catch (InterruptedException e) {
				samplerNotify.setInactive();
			}

			if (lastIteration >= 4) {
				// In repetative mode the sampler will continue to run until it is stopped or an error occurs
				ctx.nanolibAccessor.getSamplerInterface().stop(deviceHandle);
				break;
			}
		}
	}

	/**
	 * @brief Configures and starts sampling process in continuous mode with notificaion
	 */
	public void processSamplerWithNotificationContinuous() {
		System.out.println("\nSampler with notification in continuous mode:");

		configure(SamplerMode.Continuous);

		SamplerNotifyExample samplerNotify = new SamplerNotifyExample(this);

		start(samplerNotify);
		try {
			Thread.sleep(periodMilliseconds * 10);
		} catch (InterruptedException e) {
			samplerNotify.setInactive();
		}
		// In continuous the sampler will continue to run until it is stopped or an error occurs
		ctx.nanolibAccessor.getSamplerInterface().stop(deviceHandle);
	}

	/**
	 * @brief Configures the sampler
	 * 
	 * @param mode the mode to use
	 */
    private void configure(SamplerMode mode) {
        SamplerConfiguration samplerConfiguration = new SamplerConfiguration();

        OdIndexVector odIndexVector = new OdIndexVector();
        for (TrackedAddress trackedAddress : trackedAddresses) {
            odIndexVector.add(trackedAddress.odIndex);
        }
        samplerConfiguration.setTrackedAddresses(odIndexVector);

        SamplerTrigger samplerTriggerStart = new SamplerTrigger();
        samplerTriggerStart.setCondition(triggerCondition);
        samplerTriggerStart.setAddress(triggerAddress);
        samplerTriggerStart.setValue(triggerValue);

        samplerConfiguration.setStartTrigger(samplerTriggerStart);
        samplerConfiguration.setPeriodMilliseconds(periodMilliseconds);
        samplerConfiguration.setDurationMilliseconds((mode == SamplerMode.Continuous) ? 0 : 4000);
        samplerConfiguration.setMode(mode);
        samplerConfiguration.setUsingSoftwareImplementation(mode == SamplerMode.Continuous);

        ctx.nanolibAccessor.getSamplerInterface().configure(deviceHandle, samplerConfiguration);
    }

	/**
	 * @brief Start the sampler
	 * 
	 * @param samplerNotify the notify callback to use (if any)
	 */
    private void start(SamplerNotifyExample samplerNotify) {
        // Resetting state variables
        lastIteration = 0;
        sampleNumber = 0;
        headerPrinted = false;

        // Deactivate the start trigger
        ctx.nanolibAccessor.writeNumber(deviceHandle, triggerValueInactive, triggerAddress, 32);

        try {
            ctx.nanolibAccessor.getSamplerInterface().start(deviceHandle, samplerNotify, 0);
        } catch (Exception e) {
            if (samplerNotify != null) samplerNotify.setInactive();
            throw e;
        }

        // Activate start trigger
        try {
            ctx.nanolibAccessor.writeNumber(deviceHandle, triggerValueActive, triggerAddress, 32);
        } catch (Exception e) {
            ctx.nanolibAccessor.getSamplerInterface().stop(deviceHandle);
            throw e;
        }
    }

	/**
	 * @brief Get the sampler state
	 */
    private SamplerState getSamplerState() {
        return ctx.nanolibAccessor.getSamplerInterface().getState(deviceHandle).getResult();
    }

	/**
	 * @brief Get the sampled data 
	 */
    private SampleDataVector getSamplerData() {
        return ctx.nanolibAccessor.getSamplerInterface().getData(deviceHandle).getResult();
    }

	/**
	 * @brief Outputs the last occured error
	 * 
	 * @param lastError the last error from calling function
	 */
	private void handleSamplerFailed(ResultVoid lastError) {
		if (lastError == null) {
			if (this.getSamplerState() == SamplerState.Failed) {
				lastError = ctx.nanolibAccessor.getSamplerInterface().getLastError(deviceHandle);
			}
		}

		if (lastError.hasError()) {
            MenuUtils.handleErrorMessage(ctx, "Error during sampling: ", lastError.getError());
		}
	}

	/**
	 * @brief Output the sampled data
	 * 
	 * @param sampleDatas vector of sampled data points
	 */
    public void processSampledData(SampleDataVector sampleDatas) {
        final int numberOfTrackedAddresses = trackedAddresses.size();
    
        for (SampleData sampleData : sampleDatas) {
            SampledValueVector sampledValues = sampleData.getSampledValues();
            int numberOfSampledValues = sampledValues.size();
    
            assert (numberOfSampledValues % numberOfTrackedAddresses) == 0;
    
            if (lastIteration != sampleData.getIterationNumber().longValue()) {
                sampleNumber = 0;
                lastIteration = sampleData.getIterationNumber().longValue();
            }
    
            StringBuilder stringBuilder = new StringBuilder();
    
            if (!headerPrinted) {
                String horizontalLine = "------------------------------------------------------------\n";
                stringBuilder.append(horizontalLine);
                stringBuilder.append(String.format("%-10s%-10s", "Iteration", "Sample"));
    
                for (TrackedAddress trackedAddress : trackedAddresses) {
                    String addressName = "[" + trackedAddress.nameString + "]";
                    stringBuilder.append(String.format("%-14s%-8s", addressName, "Time"));
                }
    
                stringBuilder.append("\n").append(horizontalLine);
                headerPrinted = true;
            }
    
            for (int index = 0; index < numberOfSampledValues; index += numberOfTrackedAddresses) {
                stringBuilder.append(String.format("%-10d%-10d", lastIteration, sampleNumber));
    
                for (int trackedAddressIndex = 0; trackedAddressIndex < numberOfTrackedAddresses; trackedAddressIndex++) {
                    SampledValue sampledValue = sampledValues.get(index + trackedAddressIndex);
                    stringBuilder.append(String.format("%-14s%-8d", sampledValue.getValue(), sampledValue.getCollectTimeMsec()));
                }
    
                stringBuilder.append("\n");
                sampleNumber++;
            }
    
            System.out.print(stringBuilder.toString());
        }
    }
}
